using System;

using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Windows.Forms;

namespace KnihaJizd
{
    /// <summary>
    /// Třída pro práci se zvoleným automobilem
    /// </summary>
    class Automobil : XMLAutomobil
    {
        public List<Jizda> jizdy;
        public List<Tankovani> tankovani;
        private string auto;
        private XMLJizdy objektJizdy;
        private XMLTankovani objektTankovani;

        /// <summary>
        /// Konstruktor - načte potřebná data
        /// </summary>
        /// <param name="auto">Zvolený automobil</param>
        public Automobil(DirectoryInfo auto) : base(auto)
        {
            this.auto = auto.FullName;

            this.objektJizdy = new XMLJizdy(auto);
            this.jizdy = this.objektJizdy.Jizdy;

            this.objektTankovani = new XMLTankovani(auto);
            this.tankovani = this.objektTankovani.Tankovani;
        }

        /// <summary>
        /// Vrátí název automobilu
        /// </summary>
        public string Nazev
        {
            get
            {
                return this.nazev;
            }
        }

        /// <summary>
        /// Vrátí typ automobilu
        /// </summary>
        public string Typ
        {
            get
            {
                return this.typ;
            }
        }

        /// <summary>
        /// Vrátí SPZ/RZ automobilu
        /// </summary>
        public string SPZ
        {
            get
            {
                return this.spz;
            }
        }

        /// <summary>
        /// Vrátí popis paliva
        /// </summary>
        public string Palivo
        {
            get
            {
                return this.palivo;
            }
        }

        /// <summary>
        /// Vrátí objem nádrže automobilu
        /// </summary>
        public double Objem_nadrze
        {
            get
            {
                return this.objem_nadrze;
            }
        }

        /// <summary>
        /// Vloží novou jízdu
        /// </summary>
        /// <param name="jizda">Vkládaná jízda</param>
        public void vlozJizdu(Jizda jizda)
        {
            this.jizdy.Add(jizda);
            this.objektJizdy.UlozXML(this.jizdy);
        }

        /// <summary>
        /// Nahradí a uloží výčet jízd
        /// </summary>
        /// <param name="jizdy">Nový výčet jízd</param>
        public void nahradJizdy(List<Jizda> jizdy)
        {
            this.jizdy = jizdy;
            this.objektJizdy.UlozXML(this.jizdy);
        }

        /// <summary>
        /// Vrátí počet evidovaných kilometrů v jízdách
        /// </summary>
        public string JizdaEvidovanaVzdalenost
        {
            get
            {
                //rozdíl kilometrů poslední a první jízdy
                if (this.jizdy.Count > 0)
                    return (this.jizdy[this.jizdy.Count - 1].stavKoncovy - this.jizdy[0].stavPocatecni).ToString();
                else
                    return (0).ToString();
            }
        }

        /// <summary>
        /// Vrátí datum poslední jízdy
        /// </summary>
        public string JizdaDatumPosledni
        {
            get
            {
                if (this.jizdy.Count > 0)
                    return this.jizdy[this.jizdy.Count - 1].Datum.ToShortDateString();
                else
                    return "-";
            }
        }

        /// <summary>
        /// Vrátí trasu poslední jízdy
        /// </summary>
        public string JizdaTrasaPosledni
        {
            get
            {
                if (this.jizdy.Count > 0)
                    return this.jizdy[this.jizdy.Count - 1].jizdaOdkud + " - " + this.jizdy[this.jizdy.Count - 1].jizdaKam;
                else
                    return "-";
            }
        }






        /// <summary>
        /// Vloží nové tankování
        /// </summary>
        /// <param name="tankovani">Vkládané tankování</param>
        public void vlozTankovani(Tankovani tankovani)
        {
            this.tankovani.Add(tankovani);
            this.objektTankovani.UlozXML(this.tankovani);
        }

        /// <summary>
        /// Nahradí a uloží výčet tankování
        /// </summary>
        /// <param name="tankovani">Nový výčet tankování</param>
        public void nahradTankovani(List<Tankovani> tankovani)
        {
            this.tankovani = tankovani;
            this.objektTankovani.UlozXML(this.tankovani);
        }

        /// <summary>
        /// Vrátí počet evidovaných kilometrů v tankování
        /// </summary>
        public string TankovaniVzdalenostEvidovana
        {
            get
            {
                //rozdíl stavu kilometrů posledního a prvního tankování
                if (this.tankovani.Count > 0)
                    return (this.tankovani[this.tankovani.Count - 1].stavTachometru - this.tankovani[0].stavTachometru).ToString();
                else
                    return (0).ToString();
            }
        }
        /// <summary>
        /// Vrátí celkový objem natankovaného paliva
        /// </summary>
        public string TankovaniSpotrebaEvidovana
        {
            get
            {
                //součet objemu natankovaného paliva
                if (this.tankovani.Count > 1)
                {
                    double soucet = 0.0;

                    for (int i = 1; i < this.tankovani.Count; i++)
                        soucet += this.tankovani[i].palivoNabrano;

                    return soucet.ToString();
                }
                else
                    return "-";
            }
        }

        /// <summary>
        /// Vrátí datum posledního tankování
        /// </summary>
        public string TankovaniDatumPosledni
        {
            get
            {
                if (this.tankovani.Count > 0)
                    return this.tankovani[this.tankovani.Count - 1].Datum.ToShortDateString();
                else
                    return "-";
            }
        }

        /// <summary>
        /// Vrátí průměrnou spotřebu automobilu
        /// </summary>
        public string TankovaniPrumernaSpotreba
        {
            get
            {
                if (this.tankovani.Count > 1)
                    return Vypocty.PrumernaSpotreba(this.objem_nadrze, this.tankovani).ToString();
                else
                    return "-";
            }
        }


//------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

        public static void VymazAutomobil(DirectoryInfo auto)
        {
            if (MessageBox.Show("Opravdu vymazat automobil " + auto.Name + "?", "Otázka", MessageBoxButtons.YesNo, MessageBoxIcon.Question, MessageBoxDefaultButton.Button2) == DialogResult.Yes)
            {
                try
                {
                    Directory.Delete(auto.FullName, true);
                }
                catch (IOException ie)
                {
                    MessageBox.Show("Nastala chyba při mazání automobilu.\nProsím vymažte složku automobilu ručně.", "Chyba");
                    return;
                }
            }
        }

        public static void VytvorAutomobil(string nazev, string typ, string spz, string palivo, double objem_nadrze)
        {
            String cesta_cela = System.Reflection.Assembly.GetCallingAssembly().GetName().CodeBase;
            String cesta_data = cesta_cela.Substring(0, cesta_cela.LastIndexOf("\\")) + "\\data\\";

            if (Directory.Exists(cesta_data + nazev))
            {
                MessageBox.Show("Automobil stejného názvu již existuje.", "Chyba");
                return;
            }

            string nazev_dir = nazev + "\\";

            try
            {
                //složka
                Directory.CreateDirectory(cesta_data + nazev_dir);
                //soubor info.xml
                XMLAutomobil.VytvorAutomobil(cesta_data + nazev_dir, nazev, typ, spz, palivo, objem_nadrze);
                //prázdný soubor jizdy.xml
                XMLJizdy.VytvorPrazdneJizdy(cesta_data + nazev_dir);
                //prázdný soubor tankovani.xml
                XMLTankovani.VytvorPrazdneTankovani(cesta_data + nazev_dir);
            }
            catch (IOException ie)
            {
                MessageBox.Show("Nastala chyba při vytváření nového automobilu.", "Chyba");
                return;
            }
        }
    }
}
