using System;

using System.Collections.Generic;
using System.Text;
using System.Xml;
using System.IO;
using System.Windows.Forms;

namespace KnihaJizd
{
    /// <summary>
    /// Třída pro načítání jízd ze souboru jizdy.xml
    /// </summary>
    class XMLJizdy : System.Object
    {
        private XmlDocument xml;
        private string xml_soubor;
        private string auto;

        private List<Jizda> jizdy;

        /// <summary>
        /// Konstruktor
        /// </summary>
        /// <param name="auto">Načítaný automobil</param>
        public XMLJizdy(DirectoryInfo auto)
        {
            this.xml = new XmlDocument();
            this.jizdy = new List<Jizda>();

            this.auto = auto.FullName;
            this.xml_soubor = this.auto + "\\jizdy.xml";

            this.NactiXML();
        }

        /// <summary>
        /// Načtení souboru jizdy.xml
        /// </summary>
        private void NactiXML()
        {
            if (!File.Exists(xml_soubor))
            {
                this.VytvorXML();
            }

            try
            {
                this.xml.Load(this.xml_soubor);
            }
            catch (XmlException e)
            {
                MessageBox.Show("Nelze načíst jízdy z XML.", "Chyba", MessageBoxButtons.OK, MessageBoxIcon.Exclamation, MessageBoxDefaultButton.Button1);
            }

            this.ParsujXML();
        }

        /// <summary>
        /// Zpracování souboru jizdy.xml
        /// </summary>
        private void ParsujXML()
        {
            XmlNodeList uvodni = this.xml.GetElementsByTagName("polozka");

            foreach (XmlNode uzel in uvodni)
            {
                Jizda tmp = new Jizda();
                foreach (XmlNode udaje in uzel)
                {
                    switch (udaje.Name)
                    {
                        case "stav_tachometru_pocatecni":
                            tmp.stavPocatecni = int.Parse(udaje.InnerText);
                            break;

                        case "stav_tachometru_koncovy":
                            tmp.stavKoncovy = int.Parse(udaje.InnerText);
                            break;

                        case "jizda_odkud":
                            tmp.jizdaOdkud = udaje.InnerText;
                            break;

                        case "jizda_kam":
                            tmp.jizdaKam = udaje.InnerText;
                            break;

                        case "datum":
                            int rok = int.Parse(udaje.InnerText.Substring(6, 4));
                            int mesic = int.Parse(udaje.InnerText.Substring(3, 2));
                            int den = int.Parse(udaje.InnerText.Substring(0, 2));
                            tmp.Datum = new DateTime(rok, mesic, den);
                            break;

                        case "ridic":
                            tmp.Ridic = udaje.InnerText;
                            break;
                    }
                }
                this.jizdy.Add(tmp);
            }
            this.SeradVycet();
        }

        /// <summary>
        /// Uložení výčtu do souboru jizdy.xml
        /// </summary>
        /// <param name="jizdy">Výčet jízd</param>
        public void UlozXML(List<Jizda> jizdy)
        {
            jizdy.Sort(XMLJizdy.SeradJizdy);

            this.xml.DocumentElement.RemoveAll();

            foreach (Jizda jizda in jizdy)
            {
                XmlNode polozka = this.xml.CreateElement("polozka");

                XmlNode stav_tachometru_pocatecni = this.xml.CreateElement("stav_tachometru_pocatecni");
                XmlNode stav_tachometru_koncovy = this.xml.CreateElement("stav_tachometru_koncovy");
                XmlNode jizda_odkud = this.xml.CreateElement("jizda_odkud");
                XmlNode jizda_kam = this.xml.CreateElement("jizda_kam");
                XmlNode datum = this.xml.CreateElement("datum");
                XmlNode ridic = this.xml.CreateElement("ridic");

                stav_tachometru_pocatecni.InnerText = jizda.stavPocatecni.ToString();
                stav_tachometru_koncovy.InnerText = jizda.stavKoncovy.ToString();
                jizda_odkud.InnerText = jizda.jizdaOdkud;
                jizda_kam.InnerText = jizda.jizdaKam;
                datum.InnerText = jizda.Datum.ToString("dd.MM.yyyy");
                ridic.InnerText = jizda.Ridic;

                polozka.AppendChild(stav_tachometru_pocatecni);
                polozka.AppendChild(stav_tachometru_koncovy);
                polozka.AppendChild(jizda_odkud);
                polozka.AppendChild(jizda_kam);
                polozka.AppendChild(datum);
                polozka.AppendChild(ridic);

                this.xml.DocumentElement.AppendChild(polozka);
            }

            this.xml.Save(this.xml_soubor);
            this.jizdy = jizdy;
        }

        /// <summary>
        /// Seřazení výčtu
        /// </summary>
        private void SeradVycet()
        {
            this.jizdy.Sort(XMLJizdy.SeradJizdy);
        }

        /// <summary>
        /// Vytvoření prázdného souboru jizdy.xml
        /// </summary>
        private void VytvorXML()
        {
            XmlDocument tmp = new XmlDocument();
            XmlNode hlavni, koren;

            hlavni = tmp.CreateNode(XmlNodeType.XmlDeclaration, "", "");
            tmp.AppendChild(hlavni);

            koren = tmp.CreateElement("jizdy");
            tmp.AppendChild(koren);

            StreamWriter write = new StreamWriter(this.xml_soubor, false, System.Text.Encoding.UTF8);

            tmp.Save(write);
            write.Close();
        }

        /// <summary>
        /// Vrátí výčet načtených jízd
        /// </summary>
        public List<Jizda> Jizdy
        {
            get
            {
                return this.jizdy;
            }
        }

//------------------------------------------------------------------------------------------------------

        /// <summary>
        /// Vytvoří prázdný soubor jizdy.xml
        /// </summary>
        /// <param name="auto">Složka automobilu</param>
        public static void VytvorPrazdneJizdy(DirectoryInfo auto)
        {
            XmlDocument xml = new XmlDocument();

            XmlNode deklarace = xml.CreateNode(XmlNodeType.XmlDeclaration, "", "");
            XmlNode koren = xml.CreateElement("jizdy");

            xml.AppendChild(deklarace);
            xml.AppendChild(koren);

            xml.Save(auto.FullName + "\\jizdy.xml");
        }

        /// <summary>
        /// Vytvoří prázdný soubor jizdy.xml
        /// </summary>
        /// <param name="auto">Cesta ke složce automobilu</param>
        public static void VytvorPrazdneJizdy(string auto)
        {
            XmlDocument xml = new XmlDocument();

            XmlNode deklarace = xml.CreateNode(XmlNodeType.XmlDeclaration, "", "");
            XmlNode koren = xml.CreateElement("jizdy");

            xml.AppendChild(deklarace);
            xml.AppendChild(koren);

            xml.Save(auto + "jizdy.xml");
        }

        /// <summary>
        /// Porovnání dvou jízd
        /// </summary>
        /// <param name="x">První jízda</param>
        /// <param name="y">Druhá jízda</param>
        public static int SeradJizdy(Jizda x, Jizda y)
        {
            if (x.Datum < y.Datum)
            {
                return -1;
            }
            else
            {
                return +1;
            }
        }
    }
}
