using System;

using System.Collections.Generic;
using System.Text;
using System.Xml;
using System.IO;
using System.Windows.Forms;

namespace KnihaJizd
{
    /// <summary>
    /// Třída pro načítání tankování ze souboru tankovani.xml
    /// </summary>
    class XMLTankovani : System.Object
    {
        private XmlDocument xml;
        private string xml_soubor;
        private string auto;

        private List<Tankovani> tankovani;

        /// <summary>
        /// Konstruktor
        /// </summary>
        /// <param name="auto">Načítaný automobil</param>
        public XMLTankovani(DirectoryInfo auto)
        {
            this.xml = new XmlDocument();
            this.tankovani = new List<Tankovani>();

            this.auto = auto.FullName;
            this.xml_soubor = this.auto + "\\tankovani.xml";

            this.NactiXML();
        }

        /// <summary>
        /// Načtení souboru tankovani.xml
        /// </summary>
        private void NactiXML()
        {
            if (!File.Exists(xml_soubor))
            {
                this.VytvorXML();
            }

            try
            {
                this.xml.Load(this.xml_soubor);
            }
            catch (XmlException e)
            {
                MessageBox.Show("Nelze načíst jízdy z XML souboru.", "Chyba", MessageBoxButtons.OK, MessageBoxIcon.Exclamation, MessageBoxDefaultButton.Button1);
            }

            this.ParsujXML();
        }

        /// <summary>
        /// Zpracování souboru tankovani.xml
        /// </summary>
        private void ParsujXML()
        {
            XmlNodeList uvodni = this.xml.GetElementsByTagName("polozka");

            foreach (XmlNode uzel in uvodni)
            {
                Tankovani tmp = new Tankovani();
                foreach (XmlNode udaje in uzel)
                {
                    switch (udaje.Name)
                    {
                        case "stav_tachometru":
                            tmp.stavTachometru = int.Parse(udaje.InnerText);
                            break;

                        case "palivo_nabrano":
                            tmp.palivoNabrano = double.Parse(udaje.InnerText);
                            break;

                        case "cena_celkova":
                            tmp.cenaCelkova = double.Parse(udaje.InnerText);
                            break;

                        case "plna_nadrz":
                            if (udaje.InnerText.Equals("true") || udaje.InnerText.Equals("True"))
                            {
                                tmp.plnaNadrz = true;
                            }
                            else
                            {
                                tmp.plnaNadrz = false;
                            }
                            break;

                        case "datum":
                            int rok = int.Parse(udaje.InnerText.Substring(6, 4));
                            int mesic = int.Parse(udaje.InnerText.Substring(3, 2));
                            int den = int.Parse(udaje.InnerText.Substring(0, 2));
                            tmp.Datum = new DateTime(rok, mesic, den);
                            break;

                        case "ridic":
                            tmp.Ridic = udaje.InnerText;
                            break;
                    }
                }
                this.tankovani.Add(tmp);
            }
            this.SeradVycet();
        }

        /// <summary>
        /// Uložení výčtu do souboru tankovani.xml
        /// </summary>
        /// <param name="tankovani">Výčet tankování</param>
        public void UlozXML(List<Tankovani> tankovani)
        {
            tankovani.Sort(XMLTankovani.SeraditTankovani);

            this.xml.DocumentElement.RemoveAll();

            foreach (Tankovani tank in tankovani)
            {
                XmlNode polozka = this.xml.CreateElement("polozka");

                XmlNode stav_tachometru = this.xml.CreateElement("stav_tachometru");
                XmlNode palivo_nabrano = this.xml.CreateElement("palivo_nabrano");
                XmlNode cena_celkova = this.xml.CreateElement("cena_celkova");
                XmlNode plna_nadrz = this.xml.CreateElement("plna_nadrz");
                XmlNode datum = this.xml.CreateElement("datum");
                XmlNode ridic = this.xml.CreateElement("ridic");

                stav_tachometru.InnerText = tank.stavTachometru.ToString();
                palivo_nabrano.InnerText = tank.palivoNabrano.ToString();
                cena_celkova.InnerText = tank.cenaCelkova.ToString();
                plna_nadrz.InnerText = tank.plnaNadrz.ToString();
                datum.InnerText = tank.Datum.ToString("dd.MM.yyyy");
                ridic.InnerText = tank.Ridic;

                polozka.AppendChild(stav_tachometru);
                polozka.AppendChild(palivo_nabrano);
                polozka.AppendChild(cena_celkova);
                polozka.AppendChild(plna_nadrz);
                polozka.AppendChild(datum);
                polozka.AppendChild(ridic);

                this.xml.DocumentElement.AppendChild(polozka);
            }

            this.xml.Save(this.xml_soubor);
            this.tankovani = tankovani;
        }

        /// <summary>
        /// Seřazení výčtu
        /// </summary>
        private void SeradVycet()
        {
            this.tankovani.Sort(XMLTankovani.SeraditTankovani);
        }

        /// <summary>
        /// Vytvoření prázdného souboru jizdy.xml
        /// </summary>
        private void VytvorXML()
        {
            XmlDocument tmp = new XmlDocument();
            XmlNode hlavni, koren;

            hlavni = tmp.CreateNode(XmlNodeType.XmlDeclaration, "", "");
            tmp.AppendChild(hlavni);

            koren = tmp.CreateElement("tankovani");
            tmp.AppendChild(koren);

            StreamWriter write = new StreamWriter(this.xml_soubor, false, System.Text.Encoding.UTF8);

            tmp.Save(write);
            write.Close();
        }

        /// <summary>
        /// Vrátí výčet načtených tankování
        /// </summary>
        public List<Tankovani> Tankovani
        {
            get
            {
                return this.tankovani;
            }
        }

//----------------------------------------------------------------------------------------------------

        /// <summary>
        /// Vytvoří prázdný soubor tankovani.xml
        /// </summary>
        /// <param name="auto">Složka automobilu</param>
        public static void VytvorPrazdneTankovani(DirectoryInfo auto)
        {
            XmlDocument xml = new XmlDocument();

            XmlNode deklarace = xml.CreateNode(XmlNodeType.XmlDeclaration, "", "");
            XmlNode koren = xml.CreateElement("tankovani");

            xml.AppendChild(deklarace);
            xml.AppendChild(koren);

            xml.Save(auto.FullName + "\\tankovani.xml");
        }

        /// <summary>
        /// Vytvoří prázdný soubor tankovani.xml
        /// </summary>
        /// <param name="auto">Cesta ke složce automobilu</param>
        public static void VytvorPrazdneTankovani(string auto)
        {
            XmlDocument xml = new XmlDocument();

            XmlNode deklarace = xml.CreateNode(XmlNodeType.XmlDeclaration, "", "");
            XmlNode koren = xml.CreateElement("tankovani");

            xml.AppendChild(deklarace);
            xml.AppendChild(koren);

            xml.Save(auto + "tankovani.xml");
        }

        /// <summary>
        /// Porovnání dvou tankování
        /// </summary>
        /// <param name="x">První tankování</param>
        /// <param name="y">Druhé tankování</param>
        public static int SeraditTankovani(Tankovani x, Tankovani y)
        {
            if (x.Datum < y.Datum)
            {
                return -1;
            }
            else
            {
                return +1;
            }
        }

    }
}
