﻿using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Collections;

namespace TI
{
    /// <summary>
    /// Třída pro reprezentaci automatu
    /// </summary>
    public class Automat
    {
        private StreamReader reader = null;
        private string soubor = "";
        private const string TYP = "DKAR";
        
        private string typ = "";
        
        private int pocet_stavu = 0;
        private int pocet_vstupu = 0;

        private char[,] prechodova_tabulka = null;

        private char pocatecni_stav;
        private int pocet_vystupu = 0;
        private char[] koncove_stavy = null;

        /// <summary>
        /// Konstruktor pro prázdny automat
        /// </summary>
        public Automat(int pocet_stavu, int pocet_vstupu, char[,] prechodova_tabulka, char pocatecni_stav, int pocet_vystupu, char[] koncove_stavy)
        {
            this.typ = TYP;
            this.pocet_stavu = pocet_stavu;
            this.pocet_vstupu = pocet_vstupu;
            this.prechodova_tabulka = prechodova_tabulka;
            this.pocatecni_stav = pocatecni_stav;
            this.pocet_vystupu = pocet_vystupu;
            this.koncove_stavy = koncove_stavy;
        }
        
        /// <summary>
        /// Konstruktor pro načítaný automat
        /// </summary>
        /// <param name="soubor">Soubor načítaného automatu</param>
        public Automat(string soubor)
        {
            if (!Automat.OverAutomat(soubor))
            {
                throw new FormatException("Nesprávný formát TI souboru");
            }

            this.soubor = soubor;
            this.reader = new StreamReader(this.soubor);
            
            this.Nacti();
        }

        /// <summary>
        /// Destruktor
        /// </summary>
        ~Automat()
        {
            //this.reader.Dispose();
        }

        /// <summary>
        /// Načteme automat do třídy
        /// </summary>
        private void Nacti()
        {
            try
            {
                int i = 1; //iterátor na prováděnou akci
                string nactena_radka;

                //načítání souboru
                while (!this.reader.EndOfStream)
                {
                    //eliminace prázdných řádků
                    do
                    {
                        nactena_radka = this.odstranKomentare(this.reader.ReadLine());
                    }
                    while (nactena_radka.Length == 0);

                    switch (i)
                    {
                        //hlavička souboru
                        case 1:
                            this.typ = nactena_radka;
                            break;

                        //počet stavů
                        case 2:
                            this.pocet_stavu = int.Parse(nactena_radka);
                            break;

                        //počet vstupů
                        case 3:
                            this.pocet_vstupu = int.Parse(nactena_radka);
                            break;

                        //načítáme tabulku přechodů
                        case 4:
                            this.prechodova_tabulka = new char[this.pocet_stavu, this.pocet_vstupu];   //deklarace pole v paměti

                            //načtení tabulky do pole
                            for (int j = 0; j < this.pocet_stavu; j++)
                            {
                                int k = 0;
                                foreach (char c in nactena_radka.ToCharArray())
                                {
                                    if (!c.Equals(' '))
                                    {
                                        this.prechodova_tabulka[j, k] = c;
                                        k++;
                                    }
                                }
                                //při zpracování posledního řádku tabulky nenačítáme další řádek souboru
                                if (j + 1 < this.pocet_stavu)
                                    nactena_radka = this.odstranKomentare(this.reader.ReadLine());
                            }
                            break;

                        //načítáme počáteční stav
                        case 5:
                            this.pocatecni_stav = (nactena_radka.ToCharArray())[0];
                            break;

                        //načítáme počet koncových stavů
                        case 6:
                            this.pocet_vystupu = int.Parse(nactena_radka.Substring(0, 1));
                            this.koncove_stavy = new char[this.pocet_vystupu];

                            int l = 0;
                            foreach (char c in nactena_radka.Substring(1).ToCharArray())
                            {
                                if (!c.Equals(' '))
                                {
                                    this.koncove_stavy[l] = c;
                                    l++;
                                }
                            }
                            break;
                    }
                    i++;
                }
                //uzavření souboru
                this.reader.Dispose();
            }
            catch (Exception e) { throw new FormatException("Nesprávný formát TI souboru"); }
        }

        /// <summary>
        /// Odstraní komentáře a netisknutelné znaky
        /// </summary>
        /// <param name="text">Text s komentáři</param>
        /// <returns>Text bez komentářů</returns>
        private string odstranKomentare(string text)
        {
            //obsahuje komentář
            if (text.Contains("/"))
                text = text.Substring(0, text.IndexOf("/"));

            //odstranění netisknutelných znaků
            text = text.Trim();
            return text;
        }

//--------------------------------------------------------------------------------------------------------------------------        
        
        /// <summary>
        /// Ověření automatu
        /// </summary>
        /// <param name="soubor">Soubor načítaného automatu</param>
        /// <returns>TRUE pokud se jedná o podporovaný automat Rozpoznávacího deterministického konečného automatu</returns>
        public static bool OverAutomat(string soubor)
        {
            StreamReader reader;
            string nactena_radka;
            
            try
            {
                reader = new StreamReader(soubor);
            }
            //pokud není soubor přístupný, vrátí se false
            catch (IOException ioe) { return false; }

            //eliminace prázdných řádků
            do
                nactena_radka = reader.ReadLine();
            while (nactena_radka.Length == 0);

            //je načítaný automat podporovaný typ?
            if (nactena_radka.Contains(TYP))
                return true;

            return false;
        }

        /// <summary>
        /// Algoritmus pro součin dvou automatů
        /// </summary>
        /// <param name="prvni">První automat</param>
        /// <param name="druhy">Druhý automat</param>
        /// <returns>Výsledný automat</returns>
        public static Automat NasobAutomat(Automat prvni, Automat druhy)
        {
            if (prvni == null || druhy == null)
                throw new NullReferenceException("Jeden nebo oba z parametrů funkce je null");

            try
            {
                //pomocné proměnné
                string[,][] kartezka_tabulka = new string[prvni.pocet_stavu, druhy.pocet_stavu][];
                Queue fronta = new Queue();
                List<string> vynechat = new List<string>();
                int vysledny_pocet_stavu = 0;

                string vychozi_stav = prvni.pocatecni_stav.ToString() + druhy.pocatecni_stav.ToString();

                //který automat má méně vstupů
                int mensi_pocet_vstupu;
                if (druhy.pocet_vstupu < prvni.pocet_vstupu)
                    mensi_pocet_vstupu = druhy.pocet_vstupu;
                else
                    mensi_pocet_vstupu = prvni.pocet_vstupu;

                //vložení výchozího stavu
                fronta.Enqueue(vychozi_stav);
                vynechat.Add(vychozi_stav);
                string pocatecni_stav = vychozi_stav;

                //vytvoří výsledný automat v poli kartézského součinu
                while (fronta.Count > 0)
                {
                    //aktuální vrchol, ze kterého přecházíme
                    string aktualni_stav = fronta.Dequeue().ToString();
                    string novy_stav;

                    //pole kam se dostaneme s aktuálního vrcholu
                    string[] prechody = new string[mensi_pocet_vstupu];

                    //naplnění pole přechodů
                    for (int i = 0; i < mensi_pocet_vstupu; i++)
                    {
                        novy_stav = prvni.prechodova_tabulka[getPrvni(aktualni_stav), i].ToString() + druhy.prechodova_tabulka[getDruhy(aktualni_stav), i].ToString();
                        if (!vynechat.Contains(novy_stav))
                        {
                            fronta.Enqueue(novy_stav);
                            vynechat.Add(novy_stav);
                        }
                        prechody[i] = novy_stav;
                    }

                    //přiřazení pole přechodu do pole kartézského součinu
                    kartezka_tabulka[getPrvni(aktualni_stav), getDruhy(aktualni_stav)] = prechody;
                    vysledny_pocet_stavu++;
                }

                //přejmenování vrcholů & určení výstupních vrcholů
                int vrchol = 0;
                List<char> vystupy = new List<char>();
                List<char> vystupy1 = new List<char>(prvni.koncove_stavy);
                List<char> vystupy2 = new List<char>(druhy.koncove_stavy);

                for (int i = 0; i < prvni.pocet_stavu; i++)
                {
                    for (int j = 0; j < druhy.pocet_stavu; j++)
                    {
                        //na místě v tabulce je vrchol
                        if (kartezka_tabulka[i, j] != null)
                        {
                            //všechny výskyty jména aktuálního vrcholu přejmenujem na jednopísmenné
                            for (int x = 0; x < prvni.pocet_stavu; x++)
                            {
                                for (int y = 0; y < druhy.pocet_stavu; y++)
                                {
                                    //vrchol musí existovat!
                                    if (kartezka_tabulka[x, y] != null)
                                    {
                                        for (int z = 0; z < mensi_pocet_vstupu; z++)
                                        {
                                            //pokud souhlasí iterovaný vrchol s přejmenovaným, přejmenuje se
                                            if (kartezka_tabulka[x, y][z] == (getPismeno(i).ToString() + getPismeno(j).ToString()))
                                            {
                                                kartezka_tabulka[x, y][z] = getPismeno(vrchol).ToString();
                                            }
                                        }
                                    }
                                }
                            }

                            //změna počátečního stavu (pokud se shoduje)
                            if (pocatecni_stav == (getPismeno(i).ToString() + getPismeno(j).ToString()))
                                pocatecni_stav = getPismeno(vrchol).ToString();

                            //je vrchol koncový?
                            if (vystupy1.Contains(getPismeno(i)) && vystupy2.Contains(getPismeno(j)))
                                vystupy.Add(getPismeno(vrchol));

                            vrchol++;
                        }
                    }
                }

                //výsledná tabulka přechodů
                char[,] vysledna_prechodova_tabulka = new char[vysledny_pocet_stavu, mensi_pocet_vstupu];
                vrchol = 0;

                //projíždíme kartézskou tabulku po řádcích
                for (int i = 0; i < prvni.pocet_stavu; i++)
                {
                    for (int j = 0; j < druhy.pocet_stavu; j++)
                    {
                        if (kartezka_tabulka[i, j] != null)
                        {
                            for (int x = 0; x < mensi_pocet_vstupu; x++)
                            {
                                vysledna_prechodova_tabulka[vrchol, x] = kartezka_tabulka[i, j][x].ToCharArray()[0];
                            }
                            vrchol++;
                        }
                    }
                }

                return new Automat(vysledna_prechodova_tabulka.GetLength(0),
                    mensi_pocet_vstupu,
                    vysledna_prechodova_tabulka,
                    pocatecni_stav.ToCharArray()[0],
                    vystupy.Count,
                    vystupy.ToArray()
                );
            }
            catch (Exception e) { throw new FormatException("Nesprávný formát TI souboru"); return null; }

        }

        /// <summary>
        /// Uloží automat do souboru TI
        /// </summary>
        /// <param name="automat">Ukládaný automat</param>
        public static void UlozAutomat(Automat automat, string soubor)
        {
            StreamWriter writer = new StreamWriter(soubor);

            //typ automatu
            writer.WriteLine(automat.typ);

            //počet stavů
            writer.WriteLine(automat.pocet_stavu);

            //počet vstupů
            writer.WriteLine(automat.pocet_vstupu);

            //přechodová tabulka
            for (int i = 0; i < automat.pocet_stavu; i++)
            {
                for (int j = 0; j < automat.pocet_vstupu; j++)
                {
                    writer.Write(automat.prechodova_tabulka[i, j]);
                    if(j+1 < automat.pocet_vstupu)
                        writer.Write(' ');
                }
                writer.WriteLine();
            }

            //počáteční stav
            writer.WriteLine(automat.pocatecni_stav);

            //počet koncových stavů
            writer.Write(automat.pocet_vystupu);

            //koncové stavy
            for (int k = 0; k < automat.pocet_vystupu; k++)
            {
                writer.Write(' ');
                writer.Write(automat.koncove_stavy[k]);
            }

            //uzavření souboru
            writer.Dispose();
        }

        /// <summary>
        /// Vrátí první znak ze zadaného text - 65(pozice velkého A v ASCII tabulce)
        /// </summary>
        /// <param name="tmp">Zadaný text</param>
        /// <returns>Číslo odpovídající prvnímu znaku v ASCII tabulce</returns>
        private static int getPrvni(string tmp)
        {
            return (tmp.ToCharArray()[0] - 65);
        }

        /// <summary>
        /// Vrátí druhý znak ze zadaného textu - 65(pozice velkého A v ASCII tabulce)
        /// </summary>
        /// <param name="tmp">Zadaný text</param>
        /// <returns>Číslo odpovídající druhému znaku v ASCII tabulce</returns>
        private static int getDruhy(string tmp)
        {
            return (tmp.ToCharArray()[1] - 65);
        }

        /// <summary>
        /// Vrátí písmeno velké abecedy na pozici zadaného čísla + 65
        /// </summary>
        /// <param name="tmp">Číslo</param>
        /// <returns>Písmeno odpovídající v ASCII tabulce</returns>
        private static char getPismeno(int tmp)
        {
            return (char)(tmp + 65);
        }
    }
}
