﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO;

namespace TI
{
    /// <summary>
    /// GUI
    /// </summary>
    public partial class frmMain : Form
    {
        private Automat automat1;
        private Automat automat2;
        private Automat vysledek;
        private bool neulozeno;

        /// <summary>
        /// Konstruktor
        /// </summary>
        public frmMain()
        {
            InitializeComponent();

            this.Icon = TI.Properties.Resources.Icon;

            this.automat1 = null;
            this.automat2 = null;
        }

        /// <summary>
        /// Tlačítko Procházet pro první soubor
        /// </summary>
        private void btnProchazet1_Click(object sender, EventArgs e)
        {
            this.StatusLabel.Text = "Nebyl zvolen žádný soubor";
            
            //dialogové okno Otevřít soubor
            if (this.loadFile.ShowDialog() == DialogResult.Cancel)
                return;
            this.tbSoubor1.Text = this.loadFile.FileName;
        }

        private void tbSoubor1_TextChanged(object sender, EventArgs e)
        {
            this.automat1 = null;
            this.pbSoubor1.Image = null;

            //existuje vůbec soubor?
            if (File.Exists(this.tbSoubor1.Text))
            {
                //validní automat?
                if (Automat.OverAutomat(this.tbSoubor1.Text))
                {
                    this.pbSoubor1.Image = this.imageList.Images[2];
                    this.StatusLabel.Text = "Soubor " + Path.GetFileName(this.tbSoubor1.Text) + " obsahuje validní formát TI";
                    this.automat1 = new Automat(this.tbSoubor1.Text);
                }
                else
                {
                    this.pbSoubor1.Image = this.imageList.Images[3];
                    this.StatusLabel.Text = "Soubor " + Path.GetFileName(this.tbSoubor1.Text) + " neobsahuje validní formát TI";
                    this.automat1 = null;
                }

                //povolení tlačítka Násobit
                if (this.automat1 != null && this.automat2 != null)
                    this.btnNasobit.Enabled = true;
                else
                    this.btnNasobit.Enabled = false;
            }
        }

        /// <summary>
        /// Tlačítko Procházet pro druhý soubor
        /// </summary>
        private void btnProchazet2_Click(object sender, EventArgs e)
        {
            this.StatusLabel.Text = "Nebyl zvolen žádný soubor";
            
            //dialogové okno Otevřít soubor
            if (this.loadFile.ShowDialog() == DialogResult.Cancel)
                return;
            this.tbSoubor2.Text = this.loadFile.FileName;
        }

        private void tbSoubor2_TextChanged(object sender, EventArgs e)
        {
            this.automat2 = null;
            this.pbSoubor2.Image = null;

            //existuje vůbec soubor?
            if (File.Exists(this.tbSoubor2.Text))
            {
                //validní automat?
                if (Automat.OverAutomat(this.tbSoubor2.Text))
                {
                    this.pbSoubor2.Image = this.imageList.Images[2];
                    this.StatusLabel.Text = "Soubor " + Path.GetFileName(this.tbSoubor2.Text) + " obsahuje validní formát TI";
                    this.automat2 = new Automat(this.tbSoubor2.Text);
                }
                else
                {
                    this.pbSoubor2.Image = this.imageList.Images[3];
                    this.StatusLabel.Text = "Soubor " + Path.GetFileName(this.tbSoubor2.Text) + " neobsahuje validní formát TI";
                    this.automat2 = null;
                }
            }

            //povolení tlačítka Násobit
            if (this.automat1 != null && this.automat2 != null)
                this.btnNasobit.Enabled = true;
            else
                this.btnNasobit.Enabled = false;
        }

        /// <summary>
        /// Násobení automatů
        /// </summary>
        private void btnNasobit_Click(object sender, EventArgs e)
        {
            DateTime start = DateTime.Now;

            this.vysledek = null;
            this.vysledek = Automat.NasobAutomat(this.automat1, this.automat2);

            //povedlo se násobení?
            if (this.vysledek != null)
            {
                this.btnUlozit.Enabled = true;
                this.neulozeno = true;
                DateTime stop = DateTime.Now;
                TimeSpan delka_algoritmu = stop - start;
                this.StatusLabel.Text = "Násobení proběhlo v pořádku (operace trvala " + delka_algoritmu.TotalMilliseconds + " milisekund)";
            }
            else
            {
                this.StatusLabel.Text = "Funkce násobení vrátila NULL";
            }

        }

        /// <summary>
        /// Uloží výsledný automat
        /// </summary>
        private void btnUlozit_Click(object sender, EventArgs e)
        {
            this.StatusLabel.Text = "Nebyl zvolen výstupní soubor";

            //dialogové okno Uložit soubor
            if (this.saveFile.ShowDialog() == DialogResult.Cancel)
                return;

            Automat.UlozAutomat(this.vysledek, this.saveFile.FileName);
            this.neulozeno = false;

            this.StatusLabel.Text = "Soubor " + Path.GetFileName(this.saveFile.FileName) + " byl uložen";
        }

        /// <summary>
        /// Konec programu
        /// </summary>
        private void mnuKonec_Click(object sender, EventArgs e)
        {
            Application.Exit();
        }

        /// <summary>
        /// Zobrazí okno O programu
        /// </summary>
        private void mnuOProgramu_Click(object sender, EventArgs e)
        {
            (new frmAbout()).ShowDialog();
        }

        /// <summary>
        /// Zvaliduje vybraný soubor
        /// </summary>
        private void mnuValidator_Click(object sender, EventArgs e)
        {
            this.StatusLabel.Text = "Nebyl zvolen žádný soubor pro validaci";
            
            //dialogové okno Otevřít soubor
            if(this.loadFile.ShowDialog() == DialogResult.Cancel)
                return;

            string soubor = this.loadFile.FileName;

            //validní automat?
            if (Automat.OverAutomat(soubor) == true)
            {
                this.StatusLabel.Text = "TI soubor je validní";
                MessageBox.Show("Tento soubor automatu obsahuje správný formát.", "Validátor TI souborů", MessageBoxButtons.OK, MessageBoxIcon.Information);
            }
            else
            {
                this.StatusLabel.Text = "TI soubor není validní";
                MessageBox.Show("Tento soubor automatu neobsahuje správný formát.", "Validátor TI souborů", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// Optimalizuje vybraný soubor
        /// </summary>
        private void mnuOptimalizator_Click(object sender, EventArgs e)
        {
            this.StatusLabel.Text = "Nebyl zvolen žádný soubor pro optimalizaci";

            //dialogové okno Otevřít soubor
            if (this.loadFile.ShowDialog() == DialogResult.Cancel)
                return;

            string soubor = this.loadFile.FileName;

            //validní automat?
            if (Automat.OverAutomat(soubor) == true)
            {
                this.StatusLabel.Text = "Nebyl zvolen výstupní soubor";

                //dialogové okno Uložit soubor
                if (this.saveFile.ShowDialog() == DialogResult.Cancel)
                    return;

                //uložení automatu
                Automat.UlozAutomat(new Automat(soubor), this.saveFile.FileName);

                this.StatusLabel.Text = "Optimalizovaný soubor " + Path.GetFileName(this.saveFile.FileName) + " byl uložen";
            }
            else
            {
                this.StatusLabel.Text = "TI soubor není validní";
                MessageBox.Show("Tento soubor automatu neobsahuje správný formát.", "Validátor TI souborů", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// Zobrazí nápovědu
        /// </summary>
        private void mnuNapoveda_Click(object sender, EventArgs e)
        {
            MessageBox.Show("Rychlá nápověda pro Nástroj TI:\n" +
                "1. vyberte požadované automaty zapsané v souboru s příponou TI\n" + 
                "2. pokud jsou vybrané automaty ve správném formátu, zpřístupní se tlačítko Násobit!\n" + 
                "3. po stisknutí tlačítka se automaty vynásobí a výsledný automat zůstane v paměti počítače\n" + 
                "4. tlačítkem Uložit jako... jej lze uložit do TI souboru", "Nápověda", MessageBoxButtons.OK, MessageBoxIcon.Information);
        }

        /// <summary>
        /// Pokud je v paměti neuložený automat, nedovolí ukončit program
        /// </summary>
        private void frmMain_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (this.neulozeno)
                if (MessageBox.Show("V paměti počítače je výsledek násobení, který není uložen.\nOpravdu chcete ukončit program?", "Zavřít", MessageBoxButtons.YesNo, MessageBoxIcon.Question, MessageBoxDefaultButton.Button2) == DialogResult.No)
                    e.Cancel = true;
        }

        /// <summary>
        /// Změna kurzoru při Drag&Drop akci nad TextBoxy
        /// </summary>
        private void tbSoubor_DragEnter(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
                e.Effect = DragDropEffects.Link;
            else
                e.Effect = DragDropEffects.None;
        }

        /// <summary>
        /// Vložení jména souboru do TextBoxu při D&D
        /// </summary>
        private void tbSoubor_DragDrop(object sender, DragEventArgs e)
        {
            string[] tmp = (string[])e.Data.GetData(DataFormats.FileDrop, false);
            (sender as TextBox).Text = tmp[0];
        }
    }
}
